<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\StockIn;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AccountingController extends Controller
{
    public function index(Request $request)
    {
        // Check if dates are provided (filter is applied)
        // Filter is applied only if both dates are provided and not empty
        $hasFilter = $request->filled('start_date') && $request->filled('end_date');
        
        if ($hasFilter) {
            // Use provided dates (filter mode)
            $startDate = $request->get('start_date');
            $endDate = $request->get('end_date');
        } else {
            // No filter - show all data
            // Get earliest transaction date (from orders or stock in)
            $earliestOrder = Order::orderBy('sale_date', 'asc')->first();
            $earliestStockIn = StockIn::orderBy('created_at', 'asc')->first();
            
            if ($earliestOrder && $earliestStockIn) {
                $earliestDate = min(
                    $earliestOrder->sale_date instanceof \Carbon\Carbon 
                        ? $earliestOrder->sale_date->format('Y-m-d') 
                        : $earliestOrder->sale_date,
                    $earliestStockIn->created_at->format('Y-m-d')
                );
            } elseif ($earliestOrder) {
                $earliestDate = $earliestOrder->sale_date instanceof \Carbon\Carbon 
                    ? $earliestOrder->sale_date->format('Y-m-d') 
                    : $earliestOrder->sale_date;
            } elseif ($earliestStockIn) {
                $earliestDate = $earliestStockIn->created_at->format('Y-m-d');
            } else {
                // No data exists, use current month
                $earliestDate = Carbon::now()->startOfMonth()->format('Y-m-d');
            }
            
            $startDate = $earliestDate;
            $endDate = Carbon::now()->format('Y-m-d');
        }

        $startDateCarbon = Carbon::parse($startDate);
        $endDateCarbon = Carbon::parse($endDate);

        // Total Sales in date range
        $totalSales = Order::whereBetween('sale_date', [$startDate, $endDate])->sum('total');
        $totalOrders = Order::whereBetween('sale_date', [$startDate, $endDate])->count();
        $averageOrderValue = $totalOrders > 0 ? $totalSales / $totalOrders : 0;

        // Revenue breakdown
        $totalRevenue = Order::whereBetween('sale_date', [$startDate, $endDate])->sum('sub_total');
        $totalFuelCharges = Order::whereBetween('sale_date', [$startDate, $endDate])->sum('fuel_charges');
        $totalOtherCharges = Order::whereBetween('sale_date', [$startDate, $endDate])->sum('other_charges');

        // Cost of Goods Sold (COGS) - Total stock in cost
        $totalCOGS = StockIn::whereBetween('created_at', [$startDateCarbon, $endDateCarbon->endOfDay()])
            ->sum('total_price');

        // Net Profit = Total Sales - COGS (simplified, without expenses for now)
        $netProfit = $totalSales - $totalCOGS;
        $netProfitMargin = $totalSales > 0 ? ($netProfit / $totalSales) * 100 : 0;

        // Monthly comparison
        $currentMonthSales = Order::whereBetween('sale_date', [
            Carbon::now()->startOfMonth()->format('Y-m-d'),
            Carbon::now()->format('Y-m-d')
        ])->sum('total');

        // Get opening balance (balance before start date)
        // If showing all data from beginning, opening balance should be 0
        $openingBalance = $hasFilter ? $this->calculateOpeningBalance($startDate) : 0;

        // Ledger: Get all transactions (Sales and Stock In) in chronological order
        $ledger = collect();

        // Add Sales transactions (Credit/Income)
        $orders = Order::whereBetween('sale_date', [$startDate, $endDate])
            ->orderBy('sale_date', 'asc')
            ->orderBy('created_at', 'asc')
            ->get();

        foreach ($orders as $order) {
            $saleDate = $order->sale_date instanceof \Carbon\Carbon 
                ? $order->sale_date->format('Y-m-d') 
                : $order->sale_date;
            
            $ledger->push([
                'date' => $saleDate,
                'datetime' => $order->created_at 
                    ? $order->created_at->format('Y-m-d H:i:s') 
                    : ($saleDate . ' 12:00:00'),
                'type' => 'Sale',
                'description' => 'Sale #' . substr($order->id, 0, 8),
                'reference' => $order->id,
                'debit' => 0,
                'credit' => $order->total,
            ]);
        }

        // Add Stock In transactions (Debit/Expense)
        $stockIns = StockIn::with('product')
            ->whereBetween('created_at', [$startDateCarbon, $endDateCarbon->endOfDay()])
            ->orderBy('created_at', 'asc')
            ->get();

        foreach ($stockIns as $stockIn) {
            $productName = $stockIn->product->name ?? 'Unknown Product';
            $ledger->push([
                'date' => $stockIn->created_at->format('Y-m-d'),
                'datetime' => $stockIn->created_at->format('Y-m-d H:i:s'),
                'type' => 'Stock Purchase',
                'description' => 'Stock In - ' . $productName . ' (' . number_format($stockIn->quantity) . ' bottles)',
                'reference' => $stockIn->id,
                'debit' => $stockIn->total_price,
                'credit' => 0,
            ]);
        }

        // Sort by datetime
        $ledger = $ledger->sortBy('datetime')->values();

        // Calculate running balance starting from opening balance
        $runningBalance = $openingBalance;
        $ledger = $ledger->map(function ($item) use (&$runningBalance) {
            $runningBalance = $runningBalance + $item['credit'] - $item['debit'];
            $item['balance'] = $runningBalance;
            return $item;
        });

        return view('accounting.index', compact(
            'startDate',
            'endDate',
            'totalSales',
            'totalOrders',
            'averageOrderValue',
            'totalRevenue',
            'totalFuelCharges',
            'totalOtherCharges',
            'totalCOGS',
            'netProfit',
            'netProfitMargin',
            'currentMonthSales',
            'ledger',
            'openingBalance'
        ));
    }

    private function calculateOpeningBalance($startDate)
    {
        $startDateCarbon = Carbon::parse($startDate);

        // Total sales before start date
        $totalSalesBefore = Order::where('sale_date', '<', $startDate)->sum('total');
        
        // Total stock purchases before start date
        $totalStockBefore = StockIn::where('created_at', '<', $startDateCarbon->startOfDay())->sum('total_price');

        // Opening balance = Sales - Stock Purchases (assuming starting balance is 0)
        return $totalSalesBefore - $totalStockBefore;
    }
}
