<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Product;
use App\Models\StockIn;
use App\Models\StockOut;
use App\Models\OrderDetail;
use App\Models\PackSize;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index()
    {
        $today = Carbon::today();
        $thisMonth = Carbon::now()->startOfMonth();
        $lastMonth = Carbon::now()->subMonth()->startOfMonth();
        $lastMonthEnd = Carbon::now()->subMonth()->endOfMonth();

        // Sales Statistics
        $todaySales = Order::whereDate('sale_date', $today)->sum('total');
        $monthSales = Order::where('sale_date', '>=', $thisMonth)->sum('total');
        $lastMonthSales = Order::whereBetween('sale_date', [$lastMonth, $lastMonthEnd])->sum('total');
        $totalSales = Order::sum('total');

        // Orders Count
        $todayOrders = Order::whereDate('sale_date', $today)->count();
        $monthOrders = Order::where('sale_date', '>=', $thisMonth)->count();
        $totalOrders = Order::count();

        // Products Statistics
        $totalProducts = Product::where('status', 1)->count();
        $totalPackSizes = PackSize::count();

        // Low Stock Products (products with less than 5 packs available)
        $lowStockProducts = Product::where('status', 1)
            ->with('packSize')
            ->withSum('stockOut', 'quantity')
            ->get()
            ->map(function ($product) {
                $manualOut = $product->stock_out_sum_quantity ?? 0;
                $soldQty = OrderDetail::where('product_id', $product->id)->sum('qty');
                $currentStockBottles = max($manualOut - $soldQty, 0);
                $packBottleQuantity = $product->packSize?->quantity ?? 0;
                $availablePacks = $packBottleQuantity > 0 ? intdiv($currentStockBottles, $packBottleQuantity) : 0;

                $product->available_packs = $availablePacks;
                $product->current_stock_bottles = $currentStockBottles;
                return $product;
            })
            ->filter(fn($product) => $product->available_packs < 5 && $product->available_packs > 0)
            ->take(10);

        // Recent Sales (Last 10 orders)
        $recentSales = Order::orderBy('sale_date', 'desc')
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        // Top Selling Products (Last 30 days) - from order_snap
        $topProductsData = [];
        $recentOrders = Order::where('sale_date', '>=', Carbon::now()->subDays(30))->get();
        
        foreach ($recentOrders as $order) {
            $items = is_array($order->order_snap) ? $order->order_snap : json_decode($order->order_snap, true) ?? [];
            foreach ($items as $item) {
                $sku = $item['sku'] ?? null;
                $qty = $item['quantity'] ?? 0;
                if ($sku) {
                    if (!isset($topProductsData[$sku])) {
                        $topProductsData[$sku] = [
                            'product_id' => $sku,
                            'total_sold' => 0,
                            'product_name' => $item['name'] ?? 'N/A',
                        ];
                    }
                    $topProductsData[$sku]['total_sold'] += $qty;
                }
            }
        }
        
        usort($topProductsData, function($a, $b) {
            return $b['total_sold'] - $a['total_sold'];
        });
        
        $topProducts = collect(array_slice($topProductsData, 0, 5));

        // Sales Chart Data (Last 30 days)
        $chartData = [];
        for ($i = 29; $i >= 0; $i--) {
            $date = Carbon::now()->subDays($i);
            $sales = Order::whereDate('sale_date', $date)->sum('total');
            $chartData[] = [
                'date' => $date->format('Y-m-d'),
                'sales' => (float) $sales,
            ];
        }

        // Monthly Sales Comparison
        $currentMonthTotal = Order::where('sale_date', '>=', $thisMonth)->sum('total');
        $lastMonthTotal = Order::whereBetween('sale_date', [$lastMonth, $lastMonthEnd])->sum('total');
        $salesGrowth = $lastMonthTotal > 0 
            ? round((($currentMonthTotal - $lastMonthTotal) / $lastMonthTotal) * 100, 2)
            : 0;

        // Stock Statistics
        $totalStockIn = StockIn::sum('quantity');
        $totalStockOut = StockOut::sum('quantity');
        $totalStockValue = StockIn::sum('total_price');

        // Average Order Value
        $averageOrderValue = $totalOrders > 0 ? $totalSales / $totalOrders : 0;

        return view('dashboard', compact(
            'todaySales',
            'monthSales',
            'lastMonthSales',
            'totalSales',
            'todayOrders',
            'monthOrders',
            'totalOrders',
            'totalProducts',
            'totalPackSizes',
            'lowStockProducts',
            'recentSales',
            'topProducts',
            'chartData',
            'salesGrowth',
            'totalStockIn',
            'totalStockOut',
            'totalStockValue',
            'averageOrderValue'
        ));
    }
}

