<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Models\Order;
use App\Models\OrderDetail;
use App\Models\Product;
use App\Models\Inventory;
use Carbon\Carbon;

class PosController extends Controller
{
    
    public const active = 1;
    public const inactive = 0;
    /**
     * Show POS page
     */
public function create()
{
    $products = Product::where('status', static::active)
        ->with(['packSize'])
        ->withSum('stockOut', 'quantity') 
        ->get()
        ->each(function ($product) {
 
            $manualOut = $product->stock_out_sum_quantity ?? 0;

            
            $soldQty = OrderDetail::where('product_id', $product->id)->sum('qty');

            // Final available stock
            $currentStockBottles = max($manualOut - $soldQty, 0);
            $packBottleQuantity = $product->packSize?->quantity ?? 0;
            $availablePacks = $packBottleQuantity > 0 ? intdiv($currentStockBottles, $packBottleQuantity) : 0;
            $latestUnitPrice = $product->stockIn()->latest('created_at')->value('unit_price') ?? 0;

            $product->current_stock = $currentStockBottles;
            $product->available_packs = $availablePacks;
            $product->pack_bottle_quantity = $packBottleQuantity;
            $product->pack_price = $latestUnitPrice * ($packBottleQuantity ?: 1);
        })
        ->filter(fn($product) => ($product->available_packs ?? 0) > 0);
    return view('POS.create', compact('products'));
}

    /**
     * Store Order 
     */
  public function store(Request $request)
{
    $validated = $request->validate([
        'items' => ['required', 'array', 'min:1'],
        'items.*.sku' => ['required', 'uuid', 'exists:products,id'],
        'items.*.quantity' => ['required', 'integer', 'min:1'],
        'items.*.price' => ['required', 'numeric', 'min:0'],
        'items.*.pack_size_quantity' => ['nullable', 'integer', 'min:1'],
        'fuel_charges' => ['nullable', 'numeric', 'min:0'],
        'other_charges' => ['nullable', 'numeric', 'min:0'],
        'other_charges_detail' => ['nullable', 'string', 'max:255'],
        'total_amount' => ['required', 'numeric', 'min:0'],
        'sale_date' => ['required', 'string'],
    ]);

    DB::beginTransaction();

    try {
        $saleDateInput = $validated['sale_date'];
        try {
            $saleDate = Carbon::createFromFormat('d/m/Y', $saleDateInput)->format('Y-m-d');
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid date format. Please use dd/mm/yyyy.'
            ], 422);
        }

        // Subtotal and total calculation
        $subTotal = collect($validated['items'])->sum(fn($item) => $item['quantity'] * $item['price']);

        // ✅ Runtime total calculation adjustment (Fuel + Extra Charges subtract)
        $fuelCharges = $validated['fuel_charges'] ?? 0;
        $otherCharges = $validated['other_charges'] ?? 0;

        // Agar aap chahtay hain ke ye dono minus ho (jaise 100 - 10 - 10 = 80)
        $total = $subTotal - ($fuelCharges + $otherCharges);

        // Create Order Record
        $order = Order::create([
            'id' => Str::uuid(),
            'sale_date' => $saleDate,
            'order_snap' => [],
            'fuel_charges' => $fuelCharges,
            'other_charges' => $otherCharges,
            'other_charges_detail' => $validated['other_charges_detail'] ?? null,
            'sub_total' => $subTotal,
            'total' => $total,
        ]);

        $orderSnapshot = [];
        $requestedBottlesPerProduct = [];

        foreach ($validated['items'] as $item) {
            $product = Product::with('packSize')->findOrFail($item['sku']);

            $packSize = $product->packSize;
            if (! $packSize || $packSize->quantity < 1) {
                throw new \Exception("Pack size not configured for {$product->name}.");
            }

            $packQuantity = $packSize->quantity;
            $packCount = $item['quantity'];
            $bottleQuantity = $packCount * $packQuantity;

            $manualOut = $product->stockOut()->sum('quantity');
            $soldQty = OrderDetail::where('product_id', $product->id)->sum('qty');
            $availableBottles = max($manualOut - $soldQty, 0);
            $alreadyRequested = $requestedBottlesPerProduct[$product->id] ?? 0;

            if (($bottleQuantity + $alreadyRequested) > $availableBottles) {
                $availablePacks = intdiv($availableBottles, $packQuantity);
                throw new \Exception("Not enough stock for {$product->name}. Only {$availablePacks} pack(s) available.");
            }

            $requestedBottlesPerProduct[$product->id] = $alreadyRequested + $bottleQuantity;

            OrderDetail::create([
                'id' => Str::uuid(),
                'order_id' => $order->id,
                'product_id' => $product->id,
                'unit_price' => $item['price'],
                'qty' => $bottleQuantity,
                'total_price' => $packCount * $item['price'],
            ]);

            $orderSnapshot[] = [
                'sku' => $product->id,
                'name' => $product->name,
                'pack_count' => $packCount,
                'quantity' => $packCount, // backward compatibility
                'unit_price' => $item['price'],
                'price' => $item['price'],
                'total_price' => $packCount * $item['price'],
                'pack_size' => [
                    'id' => $packSize->id,
                    'name' => $packSize->name,
                    'bottle_quantity' => $packQuantity,
                ],
                'bottle_quantity' => $bottleQuantity,
            ];
        }

        $order->update(['order_snap' => $orderSnapshot]);

        DB::commit();

        return response()->json([
            'status' => 'success',
            'message' => 'Sale recorded successfully!',
            'order_id' => $order->id,
            'sale_date' => Carbon::parse($saleDate)->format('d/m/Y'),
        ], 201);

    } catch (\Throwable $e) {
        DB::rollBack();
        return response()->json([
            'status' => 'error',
            'message' => $e->getMessage(),
        ], 500);
    }
}


}
