<?php

namespace App\Http\Controllers;

use App\Models\PackSize;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class ProductController extends Controller
{
    public function index()
    {
        $products = Product::with('packSize')->orderBy('id', 'desc')->get();
        $packSizes = PackSize::orderBy('name')->get();

        return view('Product.index', compact('products', 'packSizes'));
    }

    public function create()
    {
        return view('inventories.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'image' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif,webp', 'max:2048'],
            'pack_size_id' => ['required', 'uuid', 'exists:pack_sizes,id'],
            'status' => ['required', 'boolean'],
        ]);

        $imageName = null;
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $imageName = Str::uuid() . '.' . $image->getClientOriginalExtension();
            $image->storeAs('products', $imageName, 'public');
        }

        Product::create([
            'id' => Str::uuid(),
            'name' => $validated['name'],
            'image' => $imageName,
            'pack_size_id' => $validated['pack_size_id'],
            'status' => $validated['status'],
        ]);

        return redirect('products')->with('success', 'Product added successfully!');
    }

    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'image' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif,webp', 'max:2048'],
            'pack_size_id' => ['required', 'uuid', 'exists:pack_sizes,id'],
            'status' => ['required', 'boolean'],
        ]);

        $product = Product::findOrFail($id);

        $imageName = $product->image;
        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($product->image && Storage::disk('public')->exists('products/' . $product->image)) {
                Storage::disk('public')->delete('products/' . $product->image);
            }
            
            $image = $request->file('image');
            $imageName = Str::uuid() . '.' . $image->getClientOriginalExtension();
            $image->storeAs('products', $imageName, 'public');
        }

        $product->update([
            'name' => $validated['name'],
            'image' => $imageName,
            'pack_size_id' => $validated['pack_size_id'],
            'status' => $validated['status'],
        ]);

        return redirect('products')->with('success', 'Product updated successfully!');
    }

    public function destroy($id)
    {
        Product::findOrFail($id)->delete();
        return redirect('products')->with('success', 'Product deleted successfully!');
    }
}
